//
//  Marble.m
//  Marble
//
//  Created by Jonathan Diehl on 29.11.10.
//  Copyright 2010 RWTH. All rights reserved.
//

#import "Marble.h"


#pragma mark helper methods

// reduce the absolute value of value by reduction
CGFloat areduce(CGFloat value, CGFloat reduction) {
	// positive value
	if(value > 0.0) {
		return MAX(0.0, value-reduction);
	}
	
	// negative value
	else {
		return MIN(0.0, value+reduction);
	}
}


@implementation Marble

@synthesize friction, accelerationFactor, bounce, size, bounds;
@synthesize speed;


#pragma mark init & cleanup

// designated init
- (id)initWithView:(UIView *)view;
{
	self = [super init];
	if (self != nil) {
		CGRect frame = view.bounds;

		// setup layer
		layer = [[CALayer alloc] init];
		layer.delegate = self;
		
		// default configuration
		self.friction = 0.4;
		self.accelerationFactor = 5.0;
		self.bounce = 0.8;
		self.size = 22.0;
		self.bounds = CGRectMake(frame.origin.x + 11.0, frame.origin.y + 11.0, frame.size.width - 22.0, frame.size.height - 22.0);
		
		// set position to center
		self.position = CGPointMake(frame.origin.x + (frame.size.width-size)/2.0, frame.origin.y + (frame.size.height-size)/2.0);
		
		// add layer to view
		[view.layer addSublayer:layer];
		[layer setNeedsDisplay];
		
	}
	return self;
}

// clean up
- (void) dealloc
{
	[layer removeFromSuperlayer];
	[layer release];
	[super dealloc];
}


#pragma mark CALayer drawing

- (void)drawLayer:(CALayer *)aLayer inContext:(CGContextRef)context
{
	// white background
	CGContextSetRGBFillColor(context, 1.0, 1.0, 1.0, 1.0);
	CGContextFillRect(context, aLayer.bounds);
	
	// draw circle
	CGContextSetRGBFillColor(context, 1.0, 0.0, 0.0, 1.0);
	CGContextFillEllipseInRect(context, aLayer.bounds);
}


#pragma mark accessors

- (void)setSize:(CGFloat)newSize;
{
	size = newSize;
	layer.frame = CGRectMake(0.0, 0.0, size, size);
}

- (void)setPosition:(CGPoint)position;
{
	[CATransaction begin];
	[CATransaction setValue:[NSNumber numberWithFloat:1.0/30.0] forKey:kCATransactionAnimationDuration];
	layer.position = position;
	[CATransaction commit];
}

- (CGPoint)position;
{
	return layer.position;
}


#pragma mark control methods

- (void)accelerate:(CGPoint)acceleration;
{
	speed.x += acceleration.x * accelerationFactor;
	speed.y += acceleration.y * accelerationFactor;
	
	speed.x = areduce(speed.x, friction);
	speed.y = areduce(speed.y, friction);
}

- (void)update;
{
	CGPoint position = self.position;
	
	// add speed
	position.x += speed.x;
	position.y += speed.y;
	
	// horizontal collision detection with bounds
	if(position.x < bounds.origin.x) {
		position.x = bounds.origin.x;
		speed.x = -speed.x * bounce;
	} else if(position.x > bounds.origin.x + bounds.size.width) {
		position.x = bounds.origin.x + bounds.size.width;
		speed.x = -speed.x * bounce;
	}

	// vertical collision detection with bounds
	if(position.y < bounds.origin.y) {
		position.y = bounds.origin.y;
		speed.y = -speed.y * bounce;
	} else if(position.y > bounds.origin.y + bounds.size.height) {
		position.y = bounds.origin.y + bounds.size.height;
		speed.y = -speed.y * bounce;
	}
	
	// update layer position
	self.position = position;
}

- (void)stop;
{
	speed = CGPointZero;
}


@end
